<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/helpers.php';

handleCORS();

// Add error logging
ini_set('display_errors', 0);
error_reporting(E_ALL);

try {
    $method = $_SERVER['REQUEST_METHOD'];
    $db = Database::getInstance();
    $user = requireAuth();
} catch (Exception $e) {
    errorResponse('Authentication or initialization error: ' . $e->getMessage(), 500);
}

try {
switch ($method) {
    case 'POST':
        // Only admins can add expenses
        if ($user['role'] !== 'admin') {
            errorResponse('Access denied. Admin only.', 403);
        }
        
        $data = getRequestBody();
        validateRequired($data, ['expense_date', 'category', 'amount']);
        
        $result = $db->execute(
            'INSERT INTO expenses (expense_date, category, description, amount, payment_method, reference_number, created_by) VALUES (?, ?, ?, ?, ?, ?, ?)',
            [
                $data['expense_date'],
                $data['category'],
                $data['description'] ?? null,
                $data['amount'],
                $data['payment_method'] ?? 'cash',
                $data['reference_number'] ?? null,
                $user['id']
            ]
        );
        
        successResponse([
            'id' => $result['lastInsertId'],
            'expense_date' => $data['expense_date'],
            'amount' => $data['amount']
        ], 'Expense added successfully', 201);
        break;
        
    case 'GET':
        if (isset($_GET['id'])) {
            // Get single expense
            $expense = $db->fetchOne(
                'SELECT e.*, u.username as added_by FROM expenses e LEFT JOIN users u ON e.created_by = u.id WHERE e.id = ?',
                [$_GET['id']]
            );
            
            if (!$expense) {
                errorResponse('Expense not found', 404);
            }
            
            successResponse(['expense' => $expense]);
            
        } elseif (isset($_GET['summary'])) {
            // Get expenses summary
            $period = $_GET['period'] ?? 'today';
            
            // Build date condition based on period
            if ($period === 'today') {
                $dateCondition = 'DATE(expense_date) = CURDATE()';
            } elseif ($period === 'week') {
                $dateCondition = 'expense_date >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)';
            } elseif ($period === 'month') {
                $dateCondition = 'expense_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)';
            } elseif ($period === 'year') {
                $dateCondition = 'YEAR(expense_date) = YEAR(CURDATE())';
            } else {
                $dateCondition = '1=1';
            }
            
            $summary = $db->fetchOne("
                SELECT 
                    COUNT(*) as total_expenses,
                    COALESCE(SUM(amount), 0) as total_amount,
                    COALESCE(AVG(amount), 0) as avg_amount
                FROM expenses 
                WHERE $dateCondition
            ");
            
            $byCategory = $db->fetchAll("
                SELECT 
                    category,
                    COUNT(*) as count,
                    SUM(amount) as total_amount
                FROM expenses 
                WHERE $dateCondition
                GROUP BY category
                ORDER BY total_amount DESC
            ");
            
            successResponse([
                'summary' => $summary,
                'by_category' => $byCategory
            ]);
            
        } else {
            // Get all expenses with pagination
            $page = intval($_GET['page'] ?? 1);
            $limit = intval($_GET['limit'] ?? 20);
            $offset = ($page - 1) * $limit;
            
            $conditions = [];
            $params = [];
            
            if (!empty($_GET['start_date'])) {
                $conditions[] = 'expense_date >= ?';
                $params[] = $_GET['start_date'];
            }
            if (!empty($_GET['end_date'])) {
                $conditions[] = 'expense_date <= ?';
                $params[] = $_GET['end_date'];
            }
            if (!empty($_GET['category'])) {
                $conditions[] = 'category = ?';
                $params[] = $_GET['category'];
            }
            
            $where = empty($conditions) ? '' : 'WHERE ' . implode(' AND ', $conditions);
            $queryParams = array_merge($params, [$limit, $offset]);
            
            $expenses = $db->fetchAll("
                SELECT e.*, u.username as added_by 
                FROM expenses e 
                LEFT JOIN users u ON e.created_by = u.id 
                $where 
                ORDER BY e.expense_date DESC, e.created_at DESC 
                LIMIT ? OFFSET ?
            ", $queryParams);
            
            $total = $db->fetchOne("SELECT COUNT(*) as total FROM expenses $where", $params)['total'];
            
            successResponse([
                'expenses' => $expenses,
                'pagination' => [
                    'page' => $page,
                    'limit' => $limit,
                    'total' => intval($total),
                    'pages' => ceil($total / $limit)
                ]
            ]);
        }
        break;
        
    case 'PUT':
        // Only admins can update expenses
        if ($user['role'] !== 'admin') {
            errorResponse('Access denied. Admin only.', 403);
        }
        
        if (!isset($_GET['id'])) {
            errorResponse('Expense ID required', 400);
        }
        
        $data = getRequestBody();
        
        $expense = $db->fetchOne('SELECT id FROM expenses WHERE id = ?', [$_GET['id']]);
        if (!$expense) {
            errorResponse('Expense not found', 404);
        }
        
        $updates = [];
        $params = [];
        
        if (isset($data['expense_date'])) {
            $updates[] = 'expense_date = ?';
            $params[] = $data['expense_date'];
        }
        if (isset($data['category'])) {
            $updates[] = 'category = ?';
            $params[] = $data['category'];
        }
        if (isset($data['description'])) {
            $updates[] = 'description = ?';
            $params[] = $data['description'];
        }
        if (isset($data['amount'])) {
            $updates[] = 'amount = ?';
            $params[] = $data['amount'];
        }
        if (isset($data['payment_method'])) {
            $updates[] = 'payment_method = ?';
            $params[] = $data['payment_method'];
        }
        if (isset($data['reference_number'])) {
            $updates[] = 'reference_number = ?';
            $params[] = $data['reference_number'];
        }
        
        if (empty($updates)) {
            errorResponse('No fields to update', 400);
        }
        
        $params[] = $_GET['id'];
        $db->execute(
            'UPDATE expenses SET ' . implode(', ', $updates) . ' WHERE id = ?',
            $params
        );
        
        successResponse(['id' => $_GET['id']], 'Expense updated successfully');
        break;
        
    case 'DELETE':
        // Only admins can delete expenses
        if ($user['role'] !== 'admin') {
            errorResponse('Access denied. Admin only.', 403);
        }
        
        if (!isset($_GET['id'])) {
            errorResponse('Expense ID required', 400);
        }
        
        $expense = $db->fetchOne('SELECT id FROM expenses WHERE id = ?', [$_GET['id']]);
        if (!$expense) {
            errorResponse('Expense not found', 404);
        }
        
        $db->execute('DELETE FROM expenses WHERE id = ?', [$_GET['id']]);
        
        successResponse(null, 'Expense deleted successfully');
        break;
}

errorResponse('Invalid request', 400);
} catch (Exception $e) {
    errorResponse('Server error: ' . $e->getMessage(), 500);
}
?>