<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/helpers.php';

handleCORS();

$method = $_SERVER['REQUEST_METHOD'];
$db = Database::getInstance();
$user = requireAuth();

if ($method !== 'GET') {
    errorResponse('Method not allowed', 405);
}

$type = $_GET['type'] ?? '';

switch ($type) {
    case 'sales':
        $startDate = $_GET['start_date'] ?? date('Y-m-01');
        $endDate = $_GET['end_date'] ?? date('Y-m-d');
        $groupBy = $_GET['group_by'] ?? 'day';
        
        // Get summary data
        $summary = $db->fetchOne(
            "SELECT 
                COUNT(*) as total_sales, 
                COALESCE(SUM(total_amount), 0) as total_revenue, 
                COALESCE(SUM(total_profit), 0) as total_profit,
                COALESCE(SUM((SELECT SUM(quantity) FROM sale_items WHERE sale_id = s.id)), 0) as total_items_sold
            FROM sales s 
            WHERE DATE(s.created_at) BETWEEN ? AND ?",
            [$startDate, $endDate]
        );
        
        // Get daily breakdown
        $dailyData = $db->fetchAll(
            "SELECT 
                DATE(s.created_at) as period,
                COUNT(*) as total_sales,
                COALESCE(SUM(s.total_amount), 0) as total_revenue,
                COALESCE(SUM(s.total_profit), 0) as total_profit,
                COALESCE(SUM((SELECT SUM(quantity) FROM sale_items WHERE sale_id = s.id)), 0) as total_items_sold
            FROM sales s 
            WHERE DATE(s.created_at) BETWEEN ? AND ? 
            GROUP BY DATE(s.created_at) 
            ORDER BY DATE(s.created_at) DESC",
            [$startDate, $endDate]
        );
        
        successResponse([
            'summary' => $summary,
            'report' => $dailyData,
            'period' => ['start' => $startDate, 'end' => $endDate]
        ]);
        break;
        
    case 'products':
        $startDate = $_GET['start_date'] ?? date('Y-m-01');
        $endDate = $_GET['end_date'] ?? date('Y-m-d');
        $limit = intval($_GET['limit'] ?? 10);
        
        $products = $db->fetchAll(
            "SELECT 
                p.*,
                c.name as category_name,
                s.name as supplier_name,
                COALESCE(SUM(si.quantity), 0) as total_sold,
                COALESCE(SUM(si.total_price), 0) as total_revenue,
                COALESCE(SUM(si.total_profit), 0) as total_profit
            FROM products p 
            LEFT JOIN categories c ON p.category_id = c.id 
            LEFT JOIN suppliers s ON p.supplier_id = s.id
            LEFT JOIN sale_items si ON p.id = si.product_id
            LEFT JOIN sales sa ON si.sale_id = sa.id AND DATE(sa.created_at) BETWEEN ? AND ?
            WHERE p.is_active = TRUE 
            GROUP BY p.id 
            ORDER BY total_sold DESC, total_revenue DESC 
            LIMIT ?",
            [$startDate, $endDate, $limit]
        );
        
        successResponse([
            'products' => $products,
            'period' => ['start' => $startDate, 'end' => $endDate]
        ]);
        break;
        
    case 'inventory':
    case 'stock-valuation':
        $products = $db->fetchAll(
            "SELECT 
                p.*, 
                c.name as category_name, 
                s.name as supplier_name, 
                (p.quantity_in_stock * p.selling_price) as selling_value,
                (p.quantity_in_stock * p.cost_price) as cost_value
            FROM products p 
            LEFT JOIN categories c ON p.category_id = c.id 
            LEFT JOIN suppliers s ON p.supplier_id = s.id 
            WHERE p.is_active = TRUE 
            ORDER BY p.name"
        );
        
        $summary = $db->fetchOne(
            "SELECT 
                COUNT(*) as total_products, 
                SUM(quantity_in_stock) as total_quantity, 
                SUM(quantity_in_stock * selling_price) as total_selling_value,
                SUM(quantity_in_stock * cost_price) as total_cost_value
            FROM products 
            WHERE is_active = TRUE"
        );
        
        successResponse([
            'products' => $products,
            'summary' => $summary
        ]);
        break;
        
    case 'profit':
        $startDate = $_GET['start_date'] ?? date('Y-m-01');
        $endDate = $_GET['end_date'] ?? date('Y-m-d');
        $groupBy = $_GET['group_by'] ?? 'day';
        
        $profitAnalysis = $db->fetchAll(
            "SELECT 
                DATE(created_at) as period,
                SUM(total_amount) as revenue,
                SUM(total_amount - total_profit) as cost_of_goods,
                SUM(total_profit) as gross_profit,
                CASE 
                    WHEN SUM(total_amount) > 0 
                    THEN (SUM(total_profit) / SUM(total_amount)) * 100 
                    ELSE 0 
                END as profit_margin,
                COUNT(*) as transaction_count
            FROM sales 
            WHERE DATE(created_at) BETWEEN ? AND ? 
            GROUP BY DATE(created_at) 
            ORDER BY DATE(created_at) DESC",
            [$startDate, $endDate]
        );
        
        successResponse([
            'profit_analysis' => $profitAnalysis,
            'period' => ['start' => $startDate, 'end' => $endDate]
        ]);
        break;
        
    case 'item-breakdown':
        $startDate = $_GET['start_date'] ?? date('Y-m-01');
        $endDate = $_GET['end_date'] ?? date('Y-m-d');
        $limit = intval($_GET['limit'] ?? 50);
        
        // Get detailed item breakdown
        $items = $db->fetchAll(
            "SELECT 
                si.id as sale_item_id,
                s.sale_id,
                s.created_at as sale_date,
                p.name as product_name,
                p.brand,
                p.size_ml,
                c.name as category_name,
                si.quantity,
                si.unit_price,
                si.total_price,
                si.total_profit,
                u.full_name as staff_name,
                s.customer_name
            FROM sale_items si
            JOIN sales s ON si.sale_id = s.id
            JOIN products p ON si.product_id = p.id
            LEFT JOIN categories c ON p.category_id = c.id
            LEFT JOIN users u ON s.user_id = u.id
            WHERE DATE(s.created_at) BETWEEN ? AND ?
            ORDER BY s.created_at DESC
            LIMIT ?",
            [$startDate, $endDate, $limit]
        );
        
        // Get summary
        $summary = $db->fetchOne(
            "SELECT 
                SUM(si.quantity) as total_items,
                SUM(si.total_price) as total_revenue,
                COUNT(DISTINCT s.id) as total_transactions,
                COUNT(DISTINCT si.product_id) as unique_products
            FROM sale_items si
            JOIN sales s ON si.sale_id = s.id
            WHERE DATE(s.created_at) BETWEEN ? AND ?",
            [$startDate, $endDate]
        );
        
        successResponse([
            'items' => $items,
            'summary' => $summary,
            'period' => ['start' => $startDate, 'end' => $endDate]
        ]);
        break;
        
    default:
        errorResponse('Invalid report type. Available types: sales, products, inventory, stock-valuation, profit, item-breakdown', 400);
}
?>