<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/helpers.php';

handleCORS();

$method = $_SERVER['REQUEST_METHOD'];
$db = Database::getInstance();
$user = requireAuth();

switch ($method) {
    case 'POST':
        $data = getRequestBody();
        validateRequired($data, ['items']);
        
        $db->beginTransaction();
        try {
            $saleId = 'SALE-' . substr(time(), -8);
            $totalAmount = 0;
            $totalProfit = 0;
            $saleItems = [];
            
            foreach ($data['items'] as $item) {
                $product = $db->fetchOne('SELECT id, name, selling_price, cost_price, quantity_in_stock FROM products WHERE id = ? AND is_active = TRUE', [$item['product_id']]);
                
                if (!$product) {
                    throw new Exception("Product with ID {$item['product_id']} not found");
                }
                
                if ($product['quantity_in_stock'] < $item['quantity']) {
                    throw new Exception("Insufficient stock for {$product['name']}. Available: {$product['quantity_in_stock']}, Requested: {$item['quantity']}");
                }
                
                $itemTotal = $product['selling_price'] * $item['quantity'];
                $itemProfit = ($product['selling_price'] - $product['cost_price']) * $item['quantity'];
                
                $totalAmount += $itemTotal;
                $totalProfit += $itemProfit;
                
                $saleItems[] = [
                    'product_id' => $item['product_id'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $product['selling_price'],
                    'unit_cost' => $product['cost_price'],
                    'total_price' => $itemTotal,
                    'total_profit' => $itemProfit
                ];
            }
            
            $result = $db->execute(
                'INSERT INTO sales (sale_id, user_id, total_amount, total_profit, amount_paid, balance_due, payment_status, payment_method, customer_name, customer_phone, notes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
                [
                    $saleId, 
                    $user['id'], 
                    $totalAmount, 
                    $totalProfit, 
                    $data['amount_paid'] ?? $totalAmount,
                    $data['balance_due'] ?? 0,
                    $data['payment_status'] ?? 'paid',
                    $data['payment_method'] ?? 'cash', 
                    $data['customer_name'] ?? null, 
                    $data['customer_phone'] ?? null, 
                    $data['notes'] ?? null
                ]
            );
            
            $saleDbId = $result['lastInsertId'];
            
            foreach ($saleItems as $saleItem) {
                $db->execute(
                    'INSERT INTO sale_items (sale_id, product_id, quantity, unit_price, unit_cost, total_price, total_profit) VALUES (?, ?, ?, ?, ?, ?, ?)',
                    [$saleDbId, $saleItem['product_id'], $saleItem['quantity'], $saleItem['unit_price'], $saleItem['unit_cost'], $saleItem['total_price'], $saleItem['total_profit']]
                );
                
                $db->execute('UPDATE products SET quantity_in_stock = quantity_in_stock - ? WHERE id = ?', [$saleItem['quantity'], $saleItem['product_id']]);
                
                $updated = $db->fetchOne('SELECT quantity_in_stock FROM products WHERE id = ?', [$saleItem['product_id']]);
                $newStock = $updated['quantity_in_stock'];
                $previousStock = $newStock + $saleItem['quantity'];
                
                $db->execute(
                    'INSERT INTO stock_logs (product_id, user_id, type, quantity, previous_stock, new_stock, reason, reference_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?)',
                    [$saleItem['product_id'], $user['id'], 'out', $saleItem['quantity'], $previousStock, $newStock, 'Sale', $saleDbId]
                );
            }
            
            $db->commit();
            
            successResponse([
                'id' => $saleDbId,
                'sale_id' => $saleId,
                'total_amount' => $totalAmount,
                'total_profit' => $totalProfit,
                'items' => count($saleItems)
            ], 'Sale completed successfully', 201);
            
        } catch (Exception $e) {
            $db->rollBack();
            errorResponse($e->getMessage(), 400);
        }
        break;
        
    case 'GET':
        if (isset($_GET['summary']) && $_GET['summary'] === 'dashboard') {
            $period = $_GET['period'] ?? 'today';
            
            // Build date condition based on period
            if ($period === 'today') {
                $dateCondition = 'DATE(s.created_at) = CURDATE()';
            } elseif ($period === 'week') {
                $dateCondition = 's.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)';
            } elseif ($period === 'month') {
                $dateCondition = 's.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)';
            } else {
                $dateCondition = '1=1';
            }
            
            $summary = $db->fetchOne("SELECT COUNT(*) as total_sales, COALESCE(SUM(amount_paid), 0) as total_revenue, COALESCE(SUM(total_profit), 0) as total_profit, COALESCE(AVG(amount_paid), 0) as avg_sale_amount FROM sales s WHERE $dateCondition");
            
            // Get total stock value
            $stockValue = $db->fetchOne("SELECT COALESCE(SUM(quantity_in_stock * selling_price), 0) as total_stock_value, COALESCE(SUM(quantity_in_stock * cost_price), 0) as total_cost_value, COUNT(*) as total_products FROM products WHERE is_active = TRUE");
            
            // Get expenses for the period (adjust date condition for expenses table)
            $expenseDateCondition = str_replace('s.created_at', 'expense_date', $dateCondition);
            $expenseDateCondition = str_replace('DATE(expense_date)', 'expense_date', $expenseDateCondition);
            $expenses = $db->fetchOne("SELECT COALESCE(SUM(amount), 0) as total_expenses FROM expenses WHERE $expenseDateCondition");
            
            // Merge stock value and expenses into summary
            $summary['total_stock_value'] = $stockValue['total_stock_value'];
            $summary['total_cost_value'] = $stockValue['total_cost_value'];
            $summary['total_products'] = $stockValue['total_products'];
            $summary['total_expenses'] = $expenses['total_expenses'];
            $summary['net_profit'] = $summary['total_profit'] - $expenses['total_expenses'];
            
            $topProducts = $db->fetchAll("SELECT p.name, p.brand, p.size_ml, p.size_unit, SUM(si.quantity) as total_sold, SUM(si.total_price) as total_revenue FROM sale_items si JOIN sales s ON si.sale_id = s.id JOIN products p ON si.product_id = p.id WHERE $dateCondition GROUP BY si.product_id ORDER BY total_sold DESC LIMIT 5");
            
            $recentItems = $db->fetchAll("SELECT p.name, p.brand, p.size_ml, p.size_unit, si.quantity, si.unit_price, si.total_price, s.created_at, s.sale_id, u.full_name as staff_name FROM sale_items si JOIN sales s ON si.sale_id = s.id JOIN products p ON si.product_id = p.id JOIN users u ON s.user_id = u.id WHERE $dateCondition ORDER BY s.created_at DESC, si.id DESC LIMIT 10");
            
            $paymentMethods = $db->fetchAll("SELECT payment_method, COUNT(*) as count, SUM(total_amount) as total_amount FROM sales s WHERE $dateCondition GROUP BY payment_method");
            
            successResponse([
                'summary' => $summary,
                'topProducts' => $topProducts,
                'recentItems' => $recentItems,
                'paymentMethods' => $paymentMethods
            ]);
        } elseif (isset($_GET['id'])) {
            $sale = $db->fetchOne('SELECT s.*, u.full_name as staff_name FROM sales s LEFT JOIN users u ON s.user_id = u.id WHERE s.id = ?', [$_GET['id']]);
            if (!$sale) {
                errorResponse('Sale not found', 404);
            }
            $items = $db->fetchAll('SELECT si.*, p.name as product_name, p.brand, p.size_ml, p.size_unit FROM sale_items si LEFT JOIN products p ON si.product_id = p.id WHERE si.sale_id = ?', [$_GET['id']]);
            $sale['items'] = $items;
            successResponse(['sale' => $sale]);
        } else {
            $page = intval($_GET['page'] ?? 1);
            $limit = intval($_GET['limit'] ?? 20);
            $offset = ($page - 1) * $limit;
            
            $conditions = [];
            $params = [];
            
            if (!empty($_GET['start_date'])) {
                $conditions[] = 'DATE(s.created_at) >= ?';
                $params[] = $_GET['start_date'];
            }
            if (!empty($_GET['end_date'])) {
                $conditions[] = 'DATE(s.created_at) <= ?';
                $params[] = $_GET['end_date'];
            }
            if (!empty($_GET['user_id'])) {
                $conditions[] = 's.user_id = ?';
                $params[] = $_GET['user_id'];
            }
            if (!empty($_GET['payment_method'])) {
                $conditions[] = 's.payment_method = ?';
                $params[] = $_GET['payment_method'];
            }
            
            $where = empty($conditions) ? '' : 'WHERE ' . implode(' AND ', $conditions);
            $queryParams = array_merge($params, [$limit, $offset]);
            
            $sales = $db->fetchAll("SELECT s.*, u.full_name as staff_name, COUNT(si.id) as item_count FROM sales s LEFT JOIN users u ON s.user_id = u.id LEFT JOIN sale_items si ON s.id = si.sale_id $where GROUP BY s.id ORDER BY s.created_at DESC LIMIT ? OFFSET ?", $queryParams);
            
            $total = $db->fetchOne("SELECT COUNT(DISTINCT s.id) as total FROM sales s $where", $params)['total'];
            
            successResponse([
                'sales' => $sales,
                'pagination' => [
                    'page' => $page,
                    'limit' => $limit,
                    'total' => intval($total),
                    'pages' => ceil($total / $limit)
                ]
            ]);
        }
        break;
        
    case 'PUT':
        if (!isset($_GET['id'])) {
            errorResponse('Sale ID required', 400);
        }
        
        $data = getRequestBody();
        
        // Check if sale exists
        $sale = $db->fetchOne('SELECT id FROM sales WHERE id = ?', [$_GET['id']]);
        if (!$sale) {
            errorResponse('Sale not found', 404);
        }
        
        // Build update query
        $updates = [];
        $params = [];
        
        if (isset($data['customer_name'])) {
            $updates[] = 'customer_name = ?';
            $params[] = $data['customer_name'];
        }
        if (isset($data['customer_phone'])) {
            $updates[] = 'customer_phone = ?';
            $params[] = $data['customer_phone'];
        }
        if (isset($data['notes'])) {
            $updates[] = 'notes = ?';
            $params[] = $data['notes'];
        }
        if (isset($data['payment_method'])) {
            $updates[] = 'payment_method = ?';
            $params[] = $data['payment_method'];
        }
        if (isset($data['payment_status'])) {
            $updates[] = 'payment_status = ?';
            $params[] = $data['payment_status'];
        }
        if (isset($data['amount_paid'])) {
            $updates[] = 'amount_paid = ?';
            $params[] = $data['amount_paid'];
            
            // Recalculate balance
            $saleData = $db->fetchOne('SELECT total_amount FROM sales WHERE id = ?', [$_GET['id']]);
            $balance = $saleData['total_amount'] - $data['amount_paid'];
            $updates[] = 'balance_due = ?';
            $params[] = max(0, $balance);
        }
        
        if (empty($updates)) {
            errorResponse('No fields to update', 400);
        }
        
        $params[] = $_GET['id'];
        $db->execute(
            'UPDATE sales SET ' . implode(', ', $updates) . ' WHERE id = ?',
            $params
        );
        
        successResponse(['id' => $_GET['id']], 'Sale updated successfully');
        break;
        
    case 'DELETE':
        if (!isset($_GET['id'])) {
            errorResponse('Sale ID required', 400);
        }
        
        // Check if sale exists
        $sale = $db->fetchOne('SELECT id FROM sales WHERE id = ?', [$_GET['id']]);
        if (!$sale) {
            errorResponse('Sale not found', 404);
        }
        
        $db->beginTransaction();
        try {
            // Get sale items to restore stock
            $items = $db->fetchAll('SELECT product_id, quantity FROM sale_items WHERE sale_id = ?', [$_GET['id']]);
            
            // Restore stock for each item
            foreach ($items as $item) {
                $db->execute(
                    'UPDATE products SET quantity_in_stock = quantity_in_stock + ? WHERE id = ?',
                    [$item['quantity'], $item['product_id']]
                );
                
                // Log stock restoration
                $updated = $db->fetchOne('SELECT quantity_in_stock FROM products WHERE id = ?', [$item['product_id']]);
                $newStock = $updated['quantity_in_stock'];
                $previousStock = $newStock - $item['quantity'];
                
                $db->execute(
                    'INSERT INTO stock_logs (product_id, user_id, type, quantity, previous_stock, new_stock, reason, reference_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?)',
                    [$item['product_id'], $user['id'], 'in', $item['quantity'], $previousStock, $newStock, 'Sale Deleted', $_GET['id']]
                );
            }
            
            // Delete sale items
            $db->execute('DELETE FROM sale_items WHERE sale_id = ?', [$_GET['id']]);
            
            // Delete sale
            $db->execute('DELETE FROM sales WHERE id = ?', [$_GET['id']]);
            
            $db->commit();
            successResponse(null, 'Sale deleted successfully');
            
        } catch (Exception $e) {
            $db->rollBack();
            errorResponse($e->getMessage(), 400);
        }
        break;
}

errorResponse('Invalid request', 400);
?>